
# Introduction =====

#This is a script that contains code to load common variables to
#provide consistency across the project, as well as common custom
#functions used across multiple analyses

#Load a few pacakges to get working
pacman::p_load(ghibli,tidyverse)

# Get study demographic/psych/bheavioural data ========

meta <- 
  readr::read_rds("./eLife Submission Data/sleep_study_beh_psych_demographic_data.rds")

# Get electrode location data ======

topo <- 
  readr::read_rds("./eLife Submission Data/sleep_study_eeg_locations.rds") 


# Get the electrode neighbours ========
d_neighbours = 
  readr::read_rds("./eLife Submission Data/sleep_study_eeg_neighbours.rds") 


# Set some default variables =======

#Text size for figure labels
text_size = 8

#Settings for topoplots
r_v = max(sqrt(topo$x^2 + topo$y^2))
circ_scale <- r_v * 1.02

#Default colours for plots
cols <- c("Sib" = ghibli_palette("KikiLight")[2],
          "22q" = ghibli_palette("KikiMedium")[4])


# Define useful functions =====

## Standard z score =====
zscore = function(x){
  z = (x - mean(x, na.rm = T))/sd(x, na.rm = T)
  
  return(z)
}


## Bootstrap mean ####
mu_function2 <- function(data, indices) {
  
  d   <- data[indices,]        #allows boot to select sample
  return(mean(d$power))
  
}

## Cluster Correction for frequency ======

# A function that does the cluster correction
cluster_correct = function(d,neighbours,group_var,totperm = 50,clus_alpha = 0.025, montecarlo_alpha = 0.05, formula_in){
  
  # #How many permutations should we do?
  # totperm = 50  #Really we ought to be doing a thousand or so
  # 
  # #What alpha level should we set for detecting significant electrodes to then making into clusters?
  # clus_alpha       = 0.025
  # 
  # #What should the alpha level be for our clusters themselves?
  # montecarlo_alpha = 0.05
  # 
  # # - We need a formula for our stats model
  # formula_in = as.formula(power ~ group + gender + age_eeg + (1|family))
  #
  # - We need the neighbours for each unit e.g. frequency or electrode location
  #
  # - We need to know the target column that provides the grouping for our stats
  #   this is going to be the electrode or frequency channel
  
  #Lets make things simple and rename our grouping variable
  d = 
    d |>
    rename(g_v = group_var)
  
  neighbours = 
    neighbours |>
    rename(g_v = group_var)
  
  
  
  # Make the stats  =====
  
  lme_fit = function(data,formula_in){
    
    data %>%
      group_by(g_v) %>%
      nest() %>%
      mutate(coefs = map(data,~lmerTest::lmer(formula = formula_in, data = .x) |> 
                           broom.mixed::tidy(ddf = "Kenward-Roger",effects = "fixed") |> 
                           filter(term == "group22q") |> 
                           select(statistic,p.value)))
    
    
  }
  
  
  d_fit = d |> lme_fit(formula_in = formula_in)
  
  #Fit permuted models
  lme_fit_perm = function(d_in,formula_in,totperm){
    
    d_use = 
      d_in |>
      group_by(g_v) |>
      nest() |>
      mutate(d_p = map(data,~modelr::permute(.x,totperm,group) )) |> 
      select(-data) |>
      unnest(d_p) |>
      mutate(perm_df = map(perm, as.data.frame)) 
    
    d_out = 
      d_use |> 
      mutate(coefs = furrr::future_map(perm_df, ~lmerTest::lmer(formula = formula_in, data = .x) |> 
                                         broom.mixed::tidy(ddf = "Kenward-Roger",effects = "fixed") |> 
                                         filter(term == "group22q") |> 
                                         select(statistic,p.value)))
    
    return(d_out)
    
  }
  
  d_perm = d|> lme_fit_perm(formula_in = formula_in,totperm = totperm)
  
  
  #Unnest model coefficients and the neighbours, stick together
  d_fit = 
    d_fit |>
    ungroup() |>
    unnest(coefs) |>
    select(-c(data)) |>
    left_join(neighbours,by = "g_v") |>
    rename(neighbours = data) |>
    left_join(d_perm |> 
                select(-c(perm, perm_df,.id)) |>
                unnest(coefs) |>
                group_by(g_v) |>
                nest() |>
                rename(perm_coefs = data),
              by = "g_v")
  
  
  ## Now we have to do the clustery thing =============
  
  #We can try and directly translate from the MATLAB, for loops and all
  #Or we can try and do something with map. I suspect, given that we have variables
  #that grow/change shape, we are going to have to use for loops
  
  
  
  #We look for significant clusters in the real data
  
  #Preallocation
  real_clus = vector(mode = "list",length = 2)
  
  #Define a function to get clusters
  get_clusters <- function(sig_elec){
    
    #Store the data from the first significant group
    # t = tibble(g_v        = list(sig_elec$g_v[[1]]),
    #            neighbours = sig_elec$neighbours[[1]] |> pull() |> list() )
    
    t = tibble(g_v        = list(sig_elec$g_v[[1]]),
               neighbours = sig_elec$neighbours[1] )    
    
    
    #Now we loop through each successive significant group
    if(nrow(sig_elec) > 1){
      
      for(nE1 in 2:nrow(sig_elec)){
        
        clus_found = 0
        
        #Loop through our set of frequencies we have already checked for being in a cluster
        for(nclus in 1:nrow(t)){
          
          #Compare the neighbors of this group to the label of our existing cluster
          #(starting with the first significant group)
          if(any(t$g_v[[nclus]] %in% (sig_elec$neighbours[[nE1]]))){
            
            
            
            #If we have a hit, add the details of this cluster to the existing cluster data
            t$g_v[[nclus]]        = unique(c(t$g_v[[nclus]],sig_elec$g_v[nE1]))
            t$neighbours[[nclus]] = unique(c(t$neighbours[[nclus]]  ,sig_elec$neighbours[[nE1]] ))
            
            clus_found = 1
            
            break 
            
          }
        }
        
        #If we do not have any hits (e.g. this group is not a neighbour of an existing cluster)
        #we make a new cluster
        if(clus_found == 0){
          
          num_clus = dim(t)[1]
          
          t = tibble(g_v        = c(t$g_v , sig_elec$g_v[nE1]),
                     neighbours = c(t$neighbours, sig_elec$neighbours[nE1] ))
          
        }
      }
    }
    
    #Now we have a list of putative clusters, we need to merge non-unique clusters
    
    #Preallocate
    clus_merged = vector("numeric",length = 0)
    
    #Loop through each putative cluster
    for(nclus in 1:nrow(t)){
      
      #Check if this cluster is in the merge list
      if(nclus %in% clus_merged){
        
        next
        
      }else if(nclus < nrow(t)){
        
        #Look ahead at each successive cluster
        for(nclus2 in ((nclus+1):nrow(t)) ){
          
          #Check if this successive cluster contains overlapping frequencies with our focal cluster
          if(t$g_v[[nclus]] %in% t$g_v[[nclus2]] |> any()){
            
            
            #Merge group labels
            t$g_v[[nclus]]  = unique(c(t$g_v[[nclus]],
                                       t$g_v[[nclus2]])) 
            
            #Merge sets of neighbours
            t$neighbours[[nclus]]  = unique(c(t$neighbours[[nclus]],
                                              t$neighbours[[nclus2]])) 
            
            #Log that we have done a cluster merge
            clus_merged = c(clus_merged, nclus2)              
            
          }
          
        }
      }
    }
    
    #Remove non-unique clusters
    t[clus_merged,] <- NA
    
    t = 
      t |>
      drop_na()
    
    #Add the statistics for each cluster
    t = 
      t |>
      mutate(statistic = map(g_v, ~sig_elec |> 
                               filter(g_v %in% .x ) |> 
                               pull(statistic)))  
    
    return(t)
    
    
  }
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    sig_elec = 
      d_fit |>
      select(-perm_coefs)   
    
    #Do things differently if we want positive or negative clusters
    if(nsign == 1){
      
      #Get the ID and t stat of groups with p < our significance threshold and t > 0
      sig_elec =   
        sig_elec |>
        filter(p.value < clus_alpha & statistic > 0) 
    }else{
      
      #Get the ID and t stat of groups with p < our significance threshold and t < 0
      sig_elec =   
        sig_elec |>
        filter(p.value < clus_alpha & statistic < 0) 
    }
    
    sig_elec =   
      sig_elec |>
      select(g_v,statistic,p.value,neighbours)
    
    #Now use our  get_clusters function
    if(nrow(sig_elec) > 0){
      
      real_clus[[nsign]] = get_clusters(sig_elec)
      
    }
    
  }
  
  
  
  # Now we do the same process for each permuted dataset
  perm_clus = vector(mode = "list",length = 2)
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    #Preallocate this sublist
    perm_clus[[nsign]] = vector(mode = "list",length = totperm) 
    
    #Loop through each permutation
    for (nperm in 1:totperm){
      
      #Get the data for this permutation
      d_perm = 
        d_fit |> 
        select(g_v,perm_coefs,neighbours) |>
        mutate(this_perm = map(perm_coefs, ~.x[nperm,])) |>
        unnest(this_perm) |>
        select(g_v,neighbours,statistic,p.value) |>
        ungroup()
      
      
      #Do things differently if we want positive or negative clusters
      if(nsign == 1){
        
        #Get the ID and t stat of frequencies with p < our significance threshold and t > 0
        sig_elec = 
          d_perm |>
          filter(p.value < clus_alpha & statistic > 0) |>
          select(g_v,statistic,p.value,neighbours)
        
      }else{
        
        #Get the ID of frequencies with p < our significance threshold and t < 0
        sig_elec = 
          d_perm |>
          filter(p.value < clus_alpha & statistic < 0) |>
          select(g_v,statistic,p.value,neighbours)
        
      }
      
      #if we have any significant frequencies
      if(dim(sig_elec)[1] > 0){
        
        #Store the data from the first significant group
        perm_clus[[nsign]][[nperm]] = get_clusters(sig_elec)
        
      }
    }
  }
  
  
  
  # Then we get the permutation t values
  perm_tval = vector(mode = "list",length = 2)
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    #Loop through the permutations
    for (nperm in 1:totperm){
      
      #Preallocate
      temp_tval = vector("numeric",length = 0)
      
      #Loop through all the clusters in this permutation and sign
      
      if(!is_empty(perm_clus[[nsign]][[nperm]])){
        for(nclus in 1:dim(perm_clus[[nsign]][[nperm]])[1]){
          
          temp_tval = c(temp_tval, sum(perm_clus[[nsign]][[nperm]]$statistic[[nclus]]))
          
        }
        
        #Aggregate
        if(nsign == 1){
          
          perm_tval[[nsign]] = c(perm_tval[[nsign]], max(temp_tval))
          
        }else{
          
          perm_tval[[nsign]] = c(perm_tval[[nsign]], min(temp_tval))
          
        }
      }
      
    }
  }
  
  
  # Finally we get the list of corrected clusters
  
  #Preallocate
  all_clus = vector("list",length = 0)
  nc       = 0
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    #Check we have any clusters
    if(is_empty(real_clus[[nsign]])){
      next
    }
    
    #Loop through all the clusters of this sign
    for(nclus in 1:nrow(real_clus[[nsign]])){
      
      #So what you do here is to aggregate all of the t values in the cluster (?strange)
      #but that's what it is
      
      this_tval = sum(real_clus[[nsign]]$statistic[[nclus]])
      
      if(nsign == 1){
        
        #Test if the cluster is significant
        if(mean(this_tval < perm_tval[[nsign]]) < montecarlo_alpha){
          
          #If so then increase the counter
          nc = nc + 1
          all_clus[[nc]] = tibble(type        = "pos",
                                  frequencies = real_clus[[nsign]]$g_v[nclus],
                                  statistic   = this_tval,
                                  p.value     = sum(this_tval < perm_tval[[nsign]])/totperm)
          
        }
        
      }else{
        
        if(mean(this_tval > perm_tval[[nsign]]) < montecarlo_alpha){
          
          nc = nc + 1
          all_clus[[nc]] = tibble(type        = "neg",
                                  frequencies = real_clus[[nsign]]$g_v[nclus],
                                  statistic   = this_tval,
                                  p.value     = sum(this_tval > perm_tval[[nsign]])/totperm)
          
        }    
        
      }
    }
  }
  
  
  #Now we make our output data
  if(nc > 0){
    
    p_clus =
      tibble(nc = 1:nc,clus = all_clus) |>
      unnest(clus)
    
  }else{
    
    p_clus =
      tibble(nc = numeric(),
             type = character(), 
             frequencies = list(), 
             statistic = numeric(), 
             p.value = numeric()) 
    
  }
  
  return(p_clus)
  
}


## Cluster Correction for electrode =====

# A function that does the cluster correction over electrodes
cluster_correct_electrode = function(d,neighbours,group_var,totperm = 50,clus_alpha = 0.025, montecarlo_alpha = 0.05, formula_in){
  
  # #How many permutations should we do?
  # totperm = 50  #Really we ought to be doing a thousand or so
  # 
  # #What alpha level should we set for detecting significant electrodes to then making into clusters?
  # clus_alpha       = 0.025
  # 
  # #What should the alpha level be for our clusters themselves?
  # montecarlo_alpha = 0.05
  # 
  # # - We need a formula for our stats model
  # formula_in = as.formula(power ~ group + gender + age_eeg + (1|family))
  #
  # - We need the neighbours for each unit e.g. frequency or electrode location
  #
  # - We need to know the target column that provides the grouping for our stats
  #   this is going to be the electrode or frequency channel
  
  #Lets make things simple and rename our grouping variable
  d = 
    d |>
    rename(g_v = group_var)
  
  neighbours = 
    neighbours |>
    rename(g_v = group_var)
  
  
  
  # Make the stats  =====
  
  lme_fit = function(data,formula_in){
    
    data %>%
      group_by(g_v) %>%
      nest() %>%
      mutate(coefs = map(data,~lmerTest::lmer(formula = formula_in, data = .x) |> 
                           broom.mixed::tidy(ddf = "Kenward-Roger",effects = "fixed") |> 
                           filter(term == "group22q") |> 
                           select(statistic,p.value)))
    
    
  }
  
  
  d_fit = d |> lme_fit(formula_in = formula_in)
  
  #Fit permuted models
  lme_fit_perm = function(d_in,formula_in,totperm){
    
    d_use = 
      d_in |>
      group_by(g_v) |>
      nest() |>
      mutate(d_p = map(data,~modelr::permute(.x,totperm,group) )) |> 
      select(-data) |>
      unnest(d_p) |>
      mutate(perm_df = map(perm, as.data.frame)) 
    
    d_out = 
      d_use |> 
      mutate(coefs = furrr::future_map(perm_df, ~lmerTest::lmer(formula = formula_in, data = .x) |> 
                                         broom.mixed::tidy(ddf = "Kenward-Roger",effects = "fixed") |> 
                                         filter(term == "group22q") |> 
                                         select(statistic,p.value)))
    
    return(d_out)
    
  }
  
  d_perm = d|> lme_fit_perm(formula_in = formula_in,totperm = totperm)
  
  
  #Unnest model coefficients and the neighbours, stick together
  d_fit = 
    d_fit |>
    ungroup() |>
    unnest(coefs) |>
    select(-c(data)) |>
    left_join(neighbours,by = "g_v") |>
    rename(neighbours = data) |>
    left_join(d_perm |> 
                select(-c(perm, perm_df,.id)) |>
                unnest(coefs) |>
                group_by(g_v) |>
                nest() |>
                rename(perm_coefs = data),
              by = "g_v")
  
  
  ## Now we have to do the clustery thing =============
  
  #We can try and directly translate from the MATLAB, for loops and all
  #Or we can try and do something with map. I suspect, given that we have variables
  #that grow/change shape, we are going to have to use for loops
  
  
  
  #We look for significant clusters in the real data
  
  #Preallocation
  real_clus = vector(mode = "list",length = 2)
  
  #Define a function to get clusters
  get_clusters <- function(sig_elec){
    
    #Store the data from the first significant group
    # t = tibble(g_v        = list(sig_elec$g_v[[1]]),
    #            neighbours = sig_elec$neighbours[[1]] |> pull() |> list() )
    
    t = tibble(g_v        = list(sig_elec$g_v[[1]]),
               neighbours = sig_elec$neighbours[[1]] |> pull() |> list())    
    
    
    #Now we loop through each successive significant group
    if(nrow(sig_elec) > 1){
      
      for(nE1 in 2:nrow(sig_elec)){
        
        clus_found = 0
        
        #Loop through our set of frequencies we have already checked for being in a cluster
        for(nclus in 1:nrow(t)){
          
          #Compare the neighbors of this group to the label of our existing cluster
          #(starting with the first significant group)
          if(any(t$g_v[[nclus]] %in% (sig_elec$neighbours[[nE1]] |> pull()))){
            
            
            
            #If we have a hit, add the details of this cluster to the existing cluster data
            t$g_v[[nclus]]        = unique(c(t$g_v[[nclus]],sig_elec$g_v[nE1]))
            t$neighbours[[nclus]] = unique(c(t$neighbours[[nclus]] ,sig_elec$neighbours[[nE1]] |> pull() ))
            
            clus_found = 1
            
            break 
            
          }
        }
        
        #If we do not have any hits (e.g. this group is not a neighbour of an existing cluster)
        #we make a new cluster
        if(clus_found == 0){
          
          num_clus = dim(t)[1]
          
          t = tibble(g_v        = c(t$g_v , sig_elec$g_v[nE1]),
                     neighbours = c(t$neighbours, sig_elec$neighbours[[nE1]] |> pull() |>list()))
          
        }
      }
    }
    
    #Now we have a list of putative clusters, we need to merge non-unique clusters
    
    #Preallocate
    clus_merged = vector("numeric",length = 0)
    
    #Loop through each putative cluster
    for(nclus in 1:nrow(t)){
      
      #Check if this cluster is in the merge list
      if(nclus %in% clus_merged){
        
        next
        
      }else if(nclus < nrow(t)){
        
        #Look ahead at each successive cluster
        for(nclus2 in ((nclus+1):nrow(t)) ){
          
          #Check if this successive cluster contains overlapping frequencies with our focal cluster
          if(t$g_v[[nclus]] %in% t$g_v[[nclus2]] |> any()){
            
            
            #Merge group labels
            t$g_v[[nclus]]  = unique(c(t$g_v[[nclus]],
                                       t$g_v[[nclus2]])) 
            
            #Merge sets of neighbours
            t$neighbours[[nclus]]  = unique(c(t$neighbours[[nclus]],
                                              t$neighbours[[nclus2]])) 
            
            #Log that we have done a cluster merge
            clus_merged = c(clus_merged, nclus2)              
            
          }
          
        }
      }
    }
    
    #Remove non-unique clusters
    t[clus_merged,] <- NA
    
    t = 
      t |>
      drop_na()
    
    #Add the statistics for each cluster
    t = 
      t |>
      mutate(statistic = map(g_v, ~sig_elec |> 
                               filter(g_v %in% .x ) |> 
                               pull(statistic)))  
    
    return(t)
    
    
  }
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    sig_elec = 
      d_fit |>
      select(-perm_coefs)   
    
    #Do things differently if we want positive or negative clusters
    if(nsign == 1){
      
      #Get the ID and t stat of groups with p < our significance threshold and t > 0
      sig_elec =   
        sig_elec |>
        filter(p.value < clus_alpha & statistic > 0) 
    }else{
      
      #Get the ID and t stat of groups with p < our significance threshold and t < 0
      sig_elec =   
        sig_elec |>
        filter(p.value < clus_alpha & statistic < 0) 
    }
    
    sig_elec =   
      sig_elec |>
      select(g_v,statistic,p.value,neighbours)
    
    #Now use our  get_clusters function
    if(nrow(sig_elec) > 0){
      
      real_clus[[nsign]] = get_clusters(sig_elec)
      
    }
    
  }
  
  
  
  # Now we do the same process for each permuted dataset
  perm_clus = vector(mode = "list",length = 2)
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    #Preallocate this sublist
    perm_clus[[nsign]] = vector(mode = "list",length = totperm) 
    
    #Loop through each permutation
    for (nperm in 1:totperm){
      
      #Get the data for this permutation
      d_perm = 
        d_fit |> 
        select(g_v,perm_coefs,neighbours) |>
        mutate(this_perm = map(perm_coefs, ~.x[nperm,])) |>
        unnest(this_perm) |>
        select(g_v,neighbours,statistic,p.value) |>
        ungroup()
      
      
      #Do things differently if we want positive or negative clusters
      if(nsign == 1){
        
        #Get the ID and t stat of frequencies with p < our significance threshold and t > 0
        sig_elec = 
          d_perm |>
          filter(p.value < clus_alpha & statistic > 0) |>
          select(g_v,statistic,p.value,neighbours)
        
      }else{
        
        #Get the ID of frequencies with p < our significance threshold and t < 0
        sig_elec = 
          d_perm |>
          filter(p.value < clus_alpha & statistic < 0) |>
          select(g_v,statistic,p.value,neighbours)
        
      }
      
      #if we have any significant frequencies
      if(dim(sig_elec)[1] > 0){
        
        #Store the data from the first significant group
        perm_clus[[nsign]][[nperm]] = get_clusters(sig_elec)
        
      }
    }
  }
  
  
  
  # Then we get the permutation t values
  perm_tval = vector(mode = "list",length = 2)
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    #Loop through the permutations
    for (nperm in 1:totperm){
      
      #Preallocate
      temp_tval = vector("numeric",length = 0)
      
      #Loop through all the clusters in this permutation and sign
      
      if(!is_empty(perm_clus[[nsign]][[nperm]])){
        for(nclus in 1:dim(perm_clus[[nsign]][[nperm]])[1]){
          
          temp_tval = c(temp_tval, sum(perm_clus[[nsign]][[nperm]]$statistic[[nclus]]))
          
        }
        
        #Aggregate
        if(nsign == 1){
          
          perm_tval[[nsign]] = c(perm_tval[[nsign]], max(temp_tval))
          
        }else{
          
          perm_tval[[nsign]] = c(perm_tval[[nsign]], min(temp_tval))
          
        }
      }
      
    }
  }
  
  
  # Finally we get the list of corrected clusters
  
  #Preallocate
  all_clus = vector("list",length = 0)
  nc       = 0
  
  #Loop (for positive and negative clusters)
  for (nsign in 1:2){
    
    #Check we have any clusters
    if(is_empty(real_clus[[nsign]])){
      next
    }
    
    #Loop through all the clusters of this sign
    for(nclus in 1:nrow(real_clus[[nsign]])){
      
      #So what you do here is to aggregate all of the t values in the cluster (?strange)
      #but that's what it is
      
      this_tval = sum(real_clus[[nsign]]$statistic[[nclus]])
      
      if(nsign == 1){
        
        #Test if the cluster is significant
        if(mean(this_tval < perm_tval[[nsign]]) < montecarlo_alpha){
          
          #If so then increase the counter
          nc = nc + 1
          all_clus[[nc]] = tibble(type        = "pos",
                                  electrodes  = real_clus[[nsign]]$g_v[nclus],
                                  statistic   = this_tval,
                                  p.value     = sum(this_tval < perm_tval[[nsign]])/totperm)
          
        }
        
      }else{
        
        if(mean(this_tval > perm_tval[[nsign]]) < montecarlo_alpha){
          
          nc = nc + 1
          all_clus[[nc]] = tibble(type        = "neg",
                                  electrodes  = real_clus[[nsign]]$g_v[nclus],
                                  statistic   = this_tval,
                                  p.value     = sum(this_tval > perm_tval[[nsign]])/totperm)
          
        }    
        
      }
    }
  }
  
  
  #Now we make our output data
  if(nc > 0){
    
    p_clus =
      tibble(nc = 1:nc,clus = all_clus) |>
      unnest(clus)
    
  }else{
    
    p_clus =
      tibble(nc         = numeric(),
             type       = character(), 
             electrodes = list(), 
             statistic  = numeric(), 
             p.value    = numeric()) 
    
  }
  
  return(p_clus)
  
}


## Cluster Correction for mediation analysis ======

# A function that does cluster correction for the mediation data

cluster_correct_mediation = function(d,neighbours,group_var = "electrode",totperm = 50,clus_alpha = 0.025, montecarlo_alpha = 0.05){
  
  # #How many permutations should we do?
  # totperm = 50  #Really we ought to be doing a thousand or so
  # 
  # #What alpha level should we set for detecting significant electrodes to then making into clusters?
  # clus_alpha       = 0.025
  # 
  # #What should the alpha level be for our clusters themselves?
  # montecarlo_alpha = 0.05
  # 
  # - We need the neighbours for each unit e.g. frequency or electrode location
  #
  # - We need to know the target column that provides the grouping for our stats
  #   this is going to be the electrode or frequency channel
  
  library(furrr)
  
  plan(multisession, workers = 14)
  
  set.seed(1234)
  
  # 
  # d = d_2$data[[1]]
  # d = d_2$data[[i]]
  #  neighbours = d_neighbours
  #   
  
  #Lets make things simple and rename our grouping variable
  d = 
    d |>
    rename(g_v = all_of(group_var))
  
  neighbours = 
    neighbours |>
    rename(g_v = all_of(group_var))
  
  
  #Make a function that applies the logic for model fitting for each group so we can use/compare map
  mod_fit = function(d_in,med_type,out_type){
    
    #Fit models where the outcome is the dependent variable
    
    #Fit the model with the mediator as dependent variable differently for each data type
    if(out_type == "count" ){
      
      model.y = glm(outcome_value ~ mediator_value + group + age_eeg + gender,
                    family = poisson(link = "log"),
                    data = d_in)
      
    }else if(out_type == "continuous"){
      
      
      model.y = lm(outcome_value ~ mediator_value + group + age_eeg + gender,
                   data = d_in)
      
    }else if(out_type == "bernoulli"){
      
      
      model.y = glm(outcome_value ~ mediator_value + group + age_eeg + gender,
                    family = binomial(link = "logit"),
                    data = d_in)
      
    }
    
    #Fit the model with the mediator as dependent variable differently for each data type
    if(med_type == "count" ){
      
      model.m = glm(mediator_value ~ group + age_eeg + gender,
                    family = poisson(link = "log"),
                    data = d_in)
      
    }else if(med_type == "continuous"){
      
      
      model.m = lm(mediator_value ~ group + age_eeg + gender,
                     data = d_in)
      
    }else if(med_type == "bernoulli"){
      
      
      model.m = glm(mediator_value ~ group + age_eeg + gender,
                      family = binomial(link = "logit"),
                      data = d_in)
      
    }
    
    # broom::tidy(model.y)
    # broom::tidy(model.m)
    
    #Now we fit the mediation models
    
    med.out = mediation::mediate(model.m,model.y,sims = 100, boot = FALSE,
                                 treat = "group", mediator = "mediator_value",
                                 covariates = c("age_eeg","gender"),
                                 control.value = "Sib",treat.value = "22q")
    
    #Extract the model coefficients
    
    m_table = bind_cols(tibble("Term" = c('Average Mediated Effect','Average Direct Effect','Total Effect','Proportions Mediated'),
                               "Estimate"  = c(med.out$d.avg, med.out$z.avg, med.out$tau.coef,med.out$n.avg)),
                        rbind(med.out$d.avg.ci |> as_tibble_row(),
                              med.out$z.avg.ci |> as_tibble_row(),
                              med.out$tau.ci   |> as_tibble_row(),
                              med.out$n.avg.ci |> as_tibble_row()),
                        tibble("p-value"  = c(med.out$d.avg.p, med.out$z.avg.p, med.out$tau.p,med.out$n.avg.p)))    

    return(m_table)         

  }
  
  #Fit models to the true data
  d_fit =
    d %>%
    group_by(g_v,med_type,out_type) %>%
    nest() |>
    mutate(m_table = pmap(list(d_in = data, med_type = med_type, out_type = out_type),
                          mod_fit))
  
  #Generate permuted datasets and fit models to them
  d_perm = 
    d |>
    group_by(g_v,med_type,out_type) |>
    nest() |>
    mutate(d_p = map(data,~modelr::permute(.x,totperm,group) )) |> 
    select(-data) |>
    unnest(d_p) |>
    mutate(perm_df = map(perm, as.data.frame)) |>
    select(-perm) |>
    rename(data = perm_df)
  
  #Get rid of permuted datasets with complete separation if we are using a bernoulli or count-distributed mediator
  if(d_perm |> ungroup() |> distinct(med_type) |> pull() == "bernoulli"){
    
    d_perm = 
      d_perm |>
      mutate(cs = map(data, ~.x%>%
                        group_by(group, mediator_value) %>%
                        tally() %>%
                        spread(group, n) %>%
                        pivot_longer(-mediator_value) |>
                        pull(value) |>
                        is.na() |>
                        any()))
    
    d_perm = 
      d_perm |>
      unnest(cs) |>
      filter(cs == FALSE)
    
  } else if(d_perm |> ungroup() |> distinct(med_type) |> pull() == "count") {
    
    #Find and get rid of shuffles where one group has all zero values as this causes 
    #errors in the mediation function
    d_perm = 
      d_perm |>
      mutate(cs = map(data, ~.x  |>
                        mutate(mv = map_dbl(mediator_value,~ifelse(.x > 0,1,0))) |>
                        group_by(group, mv) |>
                        tally() |>
                        spread(group, n) |>
                        pivot_longer(-mv) |>
                        pull(value) |>
                        is.na() |>
                        any()))
    
    d_perm = 
      d_perm |>
      unnest(cs) |>
      filter(cs == FALSE)    
    
    
  }
  
  #Do the same for the outcome
  if(d_perm |> ungroup() |> distinct(out_type) |> pull() == "bernoulli"){
    
    d_perm = 
      d_perm |>
      mutate(cs = map(data, ~.x%>%
                        group_by(group, outcome_value) %>%
                        tally() %>%
                        spread(group, n) %>%
                        pivot_longer(-outcome_value) |>
                        pull(value) |>
                        is.na() |>
                        any()))
    
    d_perm = 
      d_perm |>
      unnest(cs) |>
      filter(cs == FALSE)
    
  } else if(d_perm |> ungroup() |> distinct(out_type) |> pull() == "count") {
    
    #Find and get rid of shuffles where one group has all zero values as this causes 
    #errors in the mediation function
    d_perm = 
      d_perm |>
      mutate(cs = map(data, ~.x  |>
                        mutate(mv = map_dbl(outcome_value,~ifelse(.x > 0,1,0))) |>
                        group_by(group, mv) |>
                        tally() |>
                        spread(group, n) |>
                        pivot_longer(-mv) |>
                        pull(value) |>
                        is.na() |>
                        any()))
    
    d_perm = 
      d_perm |>
      unnest(cs) |>
      filter(cs == FALSE)    
    
  }

  #Fit using parallel processing via the future map/furrr package
  d_perm =
    d_perm |>
    mutate(m_table = furrr::future_pmap(list(d_in = data, med_type = med_type, out_type = out_type),
                                        mod_fit))
  
  # Make the stats  =====
  
  #Unnest model coefficients and the neighbours, stick together
  d_med = 
    d_fit |>
    ungroup() |>
    unnest(m_table) |>
    select(-c(data)) |>
    left_join(neighbours,by = "g_v") |>
    rename(neighbours = data,
           p.value    = `p-value`,
           statistic  = Estimate) |>
    
    left_join(d_perm |> 
                select(-c(data)) |>
                unnest(m_table) |>
                rename(p.value    = `p-value`,
                       statistic  = Estimate) |>
                group_by(g_v,Term) |>
                nest() |>
                rename(perm_coefs = data),
              by = c("g_v","Term")) |>
    group_by(Term) |>
    nest()
  
  
  ## Now we have to do the clustery thing =============
  
  # With mediation, we have 4 values, not just one - the direct, mediated (indirect) and total effects, plus the proportion mediated
  
  #What do we do about this, what do we cluster correct, if anything?
  #We could cluster correct all separately, that would give the most choice to the user
  
  #Define a function to get clusters
  get_clusters <- function(sig_elec){
    
    #Store the data from the first significant group
    t = tibble(g_v        = list(sig_elec$g_v[[1]]),
               neighbours = sig_elec$neighbours[[1]] |> pull() |> list())    
    
    
    #Now we loop through each successive significant group
    if(nrow(sig_elec) > 1){
      
      for(nE1 in 2:nrow(sig_elec)){
        
        clus_found = 0
        
        #Loop through our set of frequencies we have already checked for being in a cluster
        for(nclus in 1:nrow(t)){
          
          #Compare the neighbors of this group to the label of our existing cluster
          #(starting with the first significant group)
          if(any(t$g_v[[nclus]] %in% (sig_elec$neighbours[[nE1]] |> pull()))){
            
            
            #If we have a hit, add the details of this cluster to the existing cluster data
            t$g_v[[nclus]]        = unique(c(t$g_v[[nclus]],sig_elec$g_v[nE1]))
            t$neighbours[[nclus]] = unique(c(t$neighbours[[nclus]] ,sig_elec$neighbours[[nE1]] |> pull() ))
            
            clus_found = 1
            
            break 
            
          }
        }
        
        #If we do not have any hits (e.g. this group is not a neighbour of an existing cluster)
        #we make a new cluster
        if(clus_found == 0){
          
          num_clus = dim(t)[1]
          
          t = tibble(g_v        = c(t$g_v , sig_elec$g_v[nE1]),
                     neighbours = c(t$neighbours, sig_elec$neighbours[[nE1]] |> pull() |>list()))
          
        }
      }
    }
    
    #Now we have a list of putative clusters, we need to merge non-unique clusters
    
    #Preallocate
    clus_merged = vector("numeric",length = 0)
    
    #Loop through each putative cluster
    for(nclus in 1:nrow(t)){
      
      #Check if this cluster is in the merge list
      if(nclus %in% clus_merged){
        
        next
        
      }else if(nclus < nrow(t)){
        
        #Look ahead at each successive cluster
        for(nclus2 in ((nclus+1):nrow(t)) ){
          
          #Check if this successive cluster contains overlapping frequencies with our focal cluster
          if(t$g_v[[nclus]] %in% t$g_v[[nclus2]] |> any()){
            
            
            #Merge group labels
            t$g_v[[nclus]]  = unique(c(t$g_v[[nclus]],
                                       t$g_v[[nclus2]])) 
            
            #Merge sets of neighbours
            t$neighbours[[nclus]]  = unique(c(t$neighbours[[nclus]],
                                              t$neighbours[[nclus2]])) 
            
            #Log that we have done a cluster merge
            clus_merged = c(clus_merged, nclus2)              
            
          }
          
        }
      }
    }
    
    #Remove non-unique clusters
    t[clus_merged,] <- NA
    
    t = 
      t |>
      drop_na()
    
    #Add the statistics for each cluster
    t = 
      t |>
      mutate(statistic = map(g_v, ~sig_elec |> 
                               filter(g_v %in% .x ) |> 
                               pull(statistic)))  
    
    return(t)
    
    
  }
  
  #Integrate this function into another function that extracts cluster data
  find_clusters <- function(d_fit,clus_alpha = 0.025, montecarlo_alpha = 0.05){
    
    #Preallocation
    real_clus = vector(mode = "list",length = 2)
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      sig_elec = 
        d_fit |>
        select(-perm_coefs)   
      
      #Do things differently if we want positive or negative clusters
      if(nsign == 1){
        
        #Get the ID and t stat of groups with p < our significance threshold and t > 0
        sig_elec =   
          sig_elec |>
          filter(p.value < clus_alpha & statistic > 0) 
      }else{
        
        #Get the ID and t stat of groups with p < our significance threshold and t < 0
        sig_elec =   
          sig_elec |>
          filter(p.value < clus_alpha & statistic < 0) 
      }
      
      sig_elec =   
        sig_elec |>
        select(g_v,statistic,p.value,neighbours)
      
      #Now use our  get_clusters function
      if(nrow(sig_elec) > 0){
        
        real_clus[[nsign]] = get_clusters(sig_elec)
        
      }
      
    }
    
    
    
    # Now we do the same process for each permuted dataset
    perm_clus = vector(mode = "list",length = 2)
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      #Preallocate this sublist
      perm_clus[[nsign]] = vector(mode = "list",length = totperm) 
      
      #Loop through each permutation
      for (nperm in 1:totperm){
        
        #Get the data for this permutation
        d_perm = 
          d_fit |> 
          select(g_v,perm_coefs,neighbours) |>
          mutate(this_perm = map(perm_coefs, ~.x[nperm,])) |>
          unnest(this_perm) |>
          select(g_v,neighbours,statistic,p.value) |>
          ungroup()
        
        
        #Do things differently if we want positive or negative clusters
        if(nsign == 1){
          
          #Get the ID and t stat of frequencies with p < our significance threshold and t > 0
          sig_elec = 
            d_perm |>
            filter(p.value < clus_alpha & statistic > 0) |>
            select(g_v,statistic,p.value,neighbours)
          
        }else{
          
          #Get the ID of frequencies with p < our significance threshold and t < 0
          sig_elec = 
            d_perm |>
            filter(p.value < clus_alpha & statistic < 0) |>
            select(g_v,statistic,p.value,neighbours)
          
        }
        
        #if we have any significant frequencies
        if(dim(sig_elec)[1] > 0){
          
          #Store the data from the first significant group
          perm_clus[[nsign]][[nperm]] = get_clusters(sig_elec)
          
        }
      }
    }
    
    
    
    # Then we get the permutation t values
    perm_tval = vector(mode = "list",length = 2)
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      #Loop through the permutations
      for (nperm in 1:totperm){
        
        #Preallocate
        temp_tval = vector("numeric",length = 0)
        
        #Loop through all the clusters in this permutation and sign
        
        if(!is_empty(perm_clus[[nsign]][[nperm]])){
          
          for(nclus in 1:dim(perm_clus[[nsign]][[nperm]])[1]){
            
            temp_tval = c(temp_tval, sum(perm_clus[[nsign]][[nperm]]$statistic[[nclus]]))
            
          }
          
          #Aggregate
          if(nsign == 1){
            
            perm_tval[[nsign]] = c(perm_tval[[nsign]], max(temp_tval))
            
          }else{
            
            perm_tval[[nsign]] = c(perm_tval[[nsign]], min(temp_tval))
            
          }
        }else{
          
          #We need to have something if there are literally no significant clusters in the shuffled data, so in this
          #case we will set the t value to zero for the permuted data
          perm_tval[[nsign]] = c(perm_tval[[nsign]],0)
          
        }
        
      }
    }
    
    
    # Finally we get the list of corrected clusters
    
    #Preallocate
    all_clus = vector("list",length = 0)
    nc       = 0
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      #Check we have any clusters
      if(is_empty(real_clus[[nsign]])){
        next
      }
      
      #Loop through all the clusters of this sign
      for(nclus in 1:nrow(real_clus[[nsign]])){
        
        #So what you do here is to aggregate all of the t values in the cluster (?strange)
        #but that's what it is
        
        this_tval = sum(real_clus[[nsign]]$statistic[[nclus]])
        
        if(nsign == 1){
          
          #Test if the cluster is significant
          if(mean(this_tval < perm_tval[[nsign]]) < montecarlo_alpha){
            
            #If so then increase the counter
            nc = nc + 1
            all_clus[[nc]] = tibble(type        = "pos",
                                    electrodes  = real_clus[[nsign]]$g_v[nclus],
                                    statistic   = this_tval,
                                    p.value     = sum(this_tval < perm_tval[[nsign]])/totperm)
            
          }
          
        }else{
          
          if(mean(this_tval > perm_tval[[nsign]]) < montecarlo_alpha){
            
            nc = nc + 1
            all_clus[[nc]] = tibble(type        = "neg",
                                    electrodes  = real_clus[[nsign]]$g_v[nclus],
                                    statistic   = this_tval,
                                    p.value     = sum(this_tval > perm_tval[[nsign]])/totperm)
            
          }    
          
        }
      }
    }
    
    
    #Now we make our output data
    if(nc > 0){
      
      p_clus =
        tibble(nc = 1:nc,clus = all_clus) |>
        unnest(clus)
      
    }else{
      
      p_clus =
        tibble(nc         = numeric(),
               type       = character(), 
               electrodes = list(), 
               statistic  = numeric(), 
               p.value    = numeric()) 
      
    }
    
    return(p_clus)
    
  }
  
  #Map this cluster hunting onto the mediation data 
  d_clus = 
    d_med |>
    mutate(clus = map(data,find_clusters,clus_alpha = 0.025, montecarlo_alpha = 0.05))


  #Finally, return out cluster corrected datasets
  return(d_clus)
  
}


## Cluster Correction for the Behavioural Interaction Data ======

# A function that does the cluster correction for the behaviour interaction data

cluster_correct_interaction = function(d,neighbours,group_var = "electrode",totperm = 50,clus_alpha = 0.025, montecarlo_alpha = 0.05){
  
  # #How many permutations should we do?
  # totperm = 50  #Really we ought to be doing a thousand or so
  # 
  # #What alpha level should we set for detecting significant electrodes to then making into clusters?
  # clus_alpha       = 0.025
  # 
  # #What should the alpha level be for our clusters themselves?
  # montecarlo_alpha = 0.05
  # 
  # - We need the neighbours for each unit e.g. frequency or electrode location
  #
  # - We need to know the target column that provides the grouping for our stats
  #   this is going to be the electrode or frequency channel
  
  
  library(furrr)
  
  plan(multisession, workers = 14)
  
  #Lets make things simple and rename our grouping variable
  d = 
    d |>
    rename(g_v = all_of(group_var))
  
  neighbours = 
    neighbours |>
    rename(g_v = all_of(group_var))
  
  
  #Make a function that applies the logic for model fitting for each group so we can use/compare map
  mod_fit = function(d_in){
    
    
    m1 = glmer(cbind(hits_morning, 15 - hits_morning) ~ value*group + age_eeg + gender + (1|family),
               data = d_in,family = binomial())
    
    e1 = emtrends(m1, pairwise ~ group, var = "value", infer = T,
                  adjust = "bonferroni", type = "response",trans = "log")
    
    d_out = e1$contrasts %>% as_tibble() %>% rename(lower.CL = asymp.LCL,upper.CL = asymp.UCL)
    
    return(d_out)
    
  }
  
  #Fit models to the true data
  d_fit =
    d %>%
    group_by(g_v) %>%
    nest() |>
    mutate(m_table = map(data,mod_fit))
  
  
  #Generate permuted datasets and fit models to them
  d_perm = 
    d |>
    group_by(g_v) |>
    nest() |>
    mutate(d_p = map(data,~modelr::permute(.x,totperm,group) )) |> 
    select(-data) |>
    unnest(d_p) |>
    mutate(perm_df = map(perm, as.data.frame)) |>
    select(-perm) |>
    rename(data = perm_df)
  
  
  #Fit using parallel processing via the future map/furrr package
  d_perm =
    d_perm |>
    mutate(m_table = furrr::future_map(data,mod_fit))
  
  # Make the stats  =====
  
  
  #Unnest model coefficients and the neighbours, stick together
  d_med = 
    d_fit |>
    ungroup() |>
    unnest(m_table) |>
    select(-c(data)) |>
    left_join(neighbours,by = "g_v") |>
    rename(neighbours = data,
           statistic  = estimate) |>
    
    left_join(d_perm |> 
                select(-c(data)) |>
                unnest(m_table) |>
                rename(statistic  = estimate) |>
                group_by(g_v) |>
                nest() |>
                rename(perm_coefs = data),
              by = c("g_v")) 
  
  ## Now we have to do the clustery thing =============
  
  #Define a function to define clusters
  get_clusters <- function(sig_elec){
    
    #Store the data from the first significant group
    t = tibble(g_v        = list(sig_elec$g_v[[1]]),
               neighbours = sig_elec$neighbours[[1]] |> pull() |> list())    
    
    
    #Now we loop through each successive significant group
    if(nrow(sig_elec) > 1){
      
      for(nE1 in 2:nrow(sig_elec)){
        
        clus_found = 0
        
        #Loop through our set of frequencies we have already checked for being in a cluster
        for(nclus in 1:nrow(t)){
          
          #Compare the neighbors of this group to the label of our existing cluster
          #(starting with the first significant group)
          if(any(t$g_v[[nclus]] %in% (sig_elec$neighbours[[nE1]] |> pull()))){
            
            
            
            #If we have a hit, add the details of this cluster to the existing cluster data
            t$g_v[[nclus]]        = unique(c(t$g_v[[nclus]],sig_elec$g_v[nE1]))
            t$neighbours[[nclus]] = unique(c(t$neighbours[[nclus]] ,sig_elec$neighbours[[nE1]] |> pull() ))
            
            clus_found = 1
            
            break 
            
          }
        }
        
        #If we do not have any hits (e.g. this group is not a neighbour of an existing cluster)
        #we make a new cluster
        if(clus_found == 0){
          
          num_clus = dim(t)[1]
          
          t = tibble(g_v        = c(t$g_v , sig_elec$g_v[nE1]),
                     neighbours = c(t$neighbours, sig_elec$neighbours[[nE1]] |> pull() |>list()))
          
        }
      }
    }
    
    #Now we have a list of putative clusters, we need to merge non-unique clusters
    
    #Preallocate
    clus_merged = vector("numeric",length = 0)
    
    #Loop through each putative cluster
    for(nclus in 1:nrow(t)){
      
      #Check if this cluster is in the merge list
      if(nclus %in% clus_merged){
        
        next
        
      }else if(nclus < nrow(t)){
        
        #Look ahead at each successive cluster
        for(nclus2 in ((nclus+1):nrow(t)) ){
          
          #Check if this successive cluster contains overlapping frequencies with our focal cluster
          if(t$g_v[[nclus]] %in% t$g_v[[nclus2]] |> any()){
            
            
            #Merge group labels
            t$g_v[[nclus]]  = unique(c(t$g_v[[nclus]],
                                       t$g_v[[nclus2]])) 
            
            #Merge sets of neighbours
            t$neighbours[[nclus]]  = unique(c(t$neighbours[[nclus]],
                                              t$neighbours[[nclus2]])) 
            
            #Log that we have done a cluster merge
            clus_merged = c(clus_merged, nclus2)              
            
          }
          
        }
      }
    }
    
    #Remove non-unique clusters
    t[clus_merged,] <- NA
    
    t = 
      t |>
      drop_na()
    
    #Add the statistics for each cluster
    t = 
      t |>
      mutate(statistic = map(g_v, ~sig_elec |> 
                               filter(g_v %in% .x ) |> 
                               pull(statistic)))  
    
    return(t)
    
    
  }
  
  #Integrate this function into another function that extracts cluster statistics
  find_clusters <- function(d_fit,clus_alpha = 0.025, montecarlo_alpha = 0.05){
    
    #Preallocation
    real_clus = vector(mode = "list",length = 2)
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      sig_elec = 
        d_fit |>
        select(-perm_coefs)   
      
      #Do things differently if we want positive or negative clusters
      if(nsign == 1){
        
        #Get the ID and t stat of groups with p < our significance threshold and t > 0
        sig_elec =   
          sig_elec |>
          filter(p.value < clus_alpha & statistic > 0) 
      }else{
        
        #Get the ID and t stat of groups with p < our significance threshold and t < 0
        sig_elec =   
          sig_elec |>
          filter(p.value < clus_alpha & statistic < 0) 
      }
      
      sig_elec =   
        sig_elec |>
        select(g_v,statistic,p.value,neighbours)
      
      #Now use our  get_clusters function
      if(nrow(sig_elec) > 0){
        
        real_clus[[nsign]] = get_clusters(sig_elec)
        
      }
      
    }
    
    
    
    # Now we do the same process for each permuted dataset
    perm_clus = vector(mode = "list",length = 2)
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      #Preallocate this sublist
      perm_clus[[nsign]] = vector(mode = "list",length = totperm) 
      
      #Loop through each permutation
      for (nperm in 1:totperm){
        
        #Get the data for this permutation
        d_perm = 
          d_fit |> 
          select(g_v,perm_coefs,neighbours) |>
          mutate(this_perm = map(perm_coefs, ~.x[nperm,])) |>
          unnest(this_perm) |>
          select(g_v,neighbours,statistic,p.value) |>
          ungroup()
        
        
        #Do things differently if we want positive or negative clusters
        if(nsign == 1){
          
          #Get the ID and t stat of frequencies with p < our significance threshold and t > 0
          sig_elec = 
            d_perm |>
            filter(p.value < clus_alpha & statistic > 0) |>
            select(g_v,statistic,p.value,neighbours)
          
        }else{
          
          #Get the ID of frequencies with p < our significance threshold and t < 0
          sig_elec = 
            d_perm |>
            filter(p.value < clus_alpha & statistic < 0) |>
            select(g_v,statistic,p.value,neighbours)
          
        }
        
        #if we have any significant frequencies
        if(dim(sig_elec)[1] > 0){
          
          #Store the data from the first significant group
          perm_clus[[nsign]][[nperm]] = get_clusters(sig_elec)
          
        }
      }
    }
    
    
    
    # Then we get the permutation t values
    perm_tval = vector(mode = "list",length = 2)
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      #Loop through the permutations
      for (nperm in 1:totperm){
        
        #Preallocate
        temp_tval = vector("numeric",length = 0)
        
        #Loop through all the clusters in this permutation and sign
        
        if(!is_empty(perm_clus[[nsign]][[nperm]])){
          for(nclus in 1:dim(perm_clus[[nsign]][[nperm]])[1]){
            
            temp_tval = c(temp_tval, sum(perm_clus[[nsign]][[nperm]]$statistic[[nclus]]))
            
          }
          
          #Aggregate
          if(nsign == 1){
            
            perm_tval[[nsign]] = c(perm_tval[[nsign]], max(temp_tval))
            
          }else{
            
            perm_tval[[nsign]] = c(perm_tval[[nsign]], min(temp_tval))
            
          }
        }
        
      }
    }
    
    
    # Finally we get the list of corrected clusters
    
    #Preallocate
    all_clus = vector("list",length = 0)
    nc       = 0
    
    #Loop (for positive and negative clusters)
    for (nsign in 1:2){
      
      #Check we have any clusters
      if(is_empty(real_clus[[nsign]])){
        next
      }
      
      #Loop through all the clusters of this sign
      for(nclus in 1:nrow(real_clus[[nsign]])){
        
        #So what you do here is to aggregate all of the t values in the cluster (?strange)
        #but that's what it is
        
        this_tval = sum(real_clus[[nsign]]$statistic[[nclus]])
        
        if(nsign == 1){
          
          #Test if the cluster is significant
          if(mean(this_tval < perm_tval[[nsign]]) < montecarlo_alpha){
            
            #If so then increase the counter
            nc = nc + 1
            all_clus[[nc]] = tibble(type        = "pos",
                                    electrodes  = real_clus[[nsign]]$g_v[nclus],
                                    statistic   = this_tval,
                                    p.value     = sum(this_tval < perm_tval[[nsign]])/totperm)
            
          }
          
        }else{
          
          if(mean(this_tval > perm_tval[[nsign]]) < montecarlo_alpha){
            
            nc = nc + 1
            all_clus[[nc]] = tibble(type        = "neg",
                                    electrodes  = real_clus[[nsign]]$g_v[nclus],
                                    statistic   = this_tval,
                                    p.value     = sum(this_tval > perm_tval[[nsign]])/totperm)
            
          }    
          
        }
      }
    }
    
    
    #Now we make our output data
    if(nc > 0){
      
      p_clus =
        tibble(nc = 1:nc,clus = all_clus) |>
        unnest(clus)
      
    }else{
      
      p_clus =
        tibble(nc         = numeric(),
               type       = character(), 
               electrodes = list(), 
               statistic  = numeric(), 
               p.value    = numeric()) 
      
    }
    
    return(p_clus)
    
  }
  
  #Map this cluster hunting onto the interaction data 
  d_clus =
    d_med |>
    find_clusters()
  
  #Finally, return out cluster corrected datasets
  return(d_clus)
  
}







